--[[
============================================================================
Script to copy the driver files to a known location where they can
be found by various driver components.

 ***** IMPORTANT NOTE *****
>> When this script is called at install time the calling script should set bOverwriteFiles to true.
>> When this script is called at uninstall time the calling script should set bOverwriteFiles to false.
============================================================================
]]--

fDebug("[Script Start]: Copy_Driver_Files");

-- Create a folder name specific for the OS type (32-bit or 64-bit) to use later when building paths.
local OSTypeFolder = "";
if (System.Is64BitOS()) then
 	OSTypeFolder = "AMD64";
else
	OSTypeFolder = "i386";
end

-- initialize local variables
local bContinue = true;
local error = 0;
local pathCommonAppDataFolder = Shell.GetFolder(SHF_APPLICATIONDATA_COMMON);
local pathRootDriverFolder = pathCommonAppDataFolder.."\\Datacard\\XPS Card Printer";

local pathUSB = pathRootDriverFolder.."\\USB";
local pathUSBforOS = pathRootDriverFolder.."\\USB\\"..OSTypeFolder;
local pathUSBSource = "";
local pathUSBforOSSource = "";

local pathNetwork = pathRootDriverFolder.."\\network";
local pathNetworkforOS = pathRootDriverFolder.."\\network\\"..OSTypeFolder;
local pathPrintDriverSource = _SourceFolder.."\\network\\*.*";
local pathPrintDriverOSSource = _SourceFolder.."\\network\\"..OSTypeFolder.."\\*.*";

if (bOverwriteFiles ~= false) then
	bOverwriteFiles = true;
end

--[[
=======================================================
Create folders to hold the driver files.
=======================================================
]]--

if not (Folder.DoesExist(pathRootDriverFolder)) then
	fDebug("  Creating root driver folder at "..pathRootDriverFolder.."...");
	Folder.Create(pathRootDriverFolder);
	error = Application.GetLastError();
	if (error ~= 0) then
		bContinue = false;
		fDebug("  ERROR: Creating folder failed with error: "..error);
	end
end

if bContinue then
	if not (Folder.DoesExist(pathUSBforOS)) then
		fDebug("  Creating USB driver folder at "..pathUSBforOS.."...");
		Folder.Create(pathUSBforOS);
		error = Application.GetLastError();
		if (error ~= 0) then
			bContinue = false;
			fDebug("  ERROR: Creating folder at failed with error: "..error);
		end
	end
end

if bContinue then
	if not (Folder.DoesExist(pathNetworkforOS)) then
		fDebug("  Creating network driver folder at "..pathNetworkforOS.."...");
		Folder.Create(pathNetworkforOS);
		if (error ~= 0) then
			bContinue = false;
			fDebug("  ERROR: Creating folder failed with error: "..error);
		end
	end
end

--[[
=======================================================
   Copy the OS specific dpinst file
=======================================================
]]--

if bContinue then
	fDebug("  Copying driver files to "..pathRootDriverFolder.."...");
	if (System.Is64BitOS()) then
		File.Copy(_SourceFolder.."\\AutoPlay\\dpinst\\dpinst64.exe", pathRootDriverFolder, false, bOverwriteFiles, true, true, nil);
	else
		File.Copy(_SourceFolder.."\\AutoPlay\\dpinst\\dpinst32.exe", pathRootDriverFolder, false, bOverwriteFiles, true, true, nil);
	end
	error = Application.GetLastError();
	if (error ~= 0) then
		bContinue = false;
		fDebug("ERROR: dpinst file copy to "..pathRootDriverFolder.." failed with error: "..error);
	end
end

--[[
=======================================================
  Copy USBLAN driver files
=======================================================
]]--
if bContinue then
	local OSVersionTable = OSUtils.GetVersionElements();
	-- any version less than 6.1 is prevented from installing so no need to check for OS's below Windows 7
	if (OSVersionTable.Major < 10) then -- All OS's older than Windows 10
		pathUSBSource =  _SourceFolder.."\\USB\\Windows7\\*.*";       
		pathUSBforOSSource = _SourceFolder.."\\USB\\Windows7\\"..OSTypeFolder.."\\*.*";
	else -- Windows 10 and beyond
		pathUSBSource = _SourceFolder.."\\USB\\Windows10\\*.*"; 
		pathUSBforOSSource = _SourceFolder.."\\USB\\Windows10\\"..OSTypeFolder.."\\*.*";
	end
	fDebug("  INFO: Copy USBLAN driver files from: "..pathUSBSource.."    and: "..pathUSBforOSSource);
	-- Copy the OS independent USB files.
	File.Copy(pathUSBSource, pathUSB, false, bOverwriteFiles, true, true, nil);
	error = Application.GetLastError();
	if (error ~= 0) then
		bContinue = false;
		fDebug("ERROR: Driver file copy to "..pathUSB.." failed with error: "..error);
	end
end	

if bContinue then
	-- Copy OS specific USB files
	File.Copy(pathUSBforOSSource, pathUSBforOS, false, bOverwriteFiles, true, true, nil);
	error = Application.GetLastError();
	if (error ~= 0) then
		bContinue = false;
		fDebug("ERROR: Driver file copy to "..pathUSBforOS.." failed with error: "..error);
	end
end

--[[
=======================================================
 Copy printer driver files 
=======================================================
]]--
-- Copy the OS independent network files.
if bContinue then
	fDebug("  INFO: Copy printer driver files from: "..pathPrintDriverSource.."    and: "..pathPrintDriverOSSource);
	File.Copy(_SourceFolder.."\\network\\*.*", pathNetwork, false, bOverwriteFiles, true, true, nil);
	error = Application.GetLastError();
	if (error ~= 0) then
		bContinue = false;
		fDebug("  ERROR: printer driver files copy to "..pathNetwork.." failed with error: "..error);
	end
end

-- Copy OS x86/x64 specific network files
if bContinue then
	File.Copy(pathPrintDriverOSSource, pathNetworkforOS, false, bOverwriteFiles, true, true, nil);
	error = Application.GetLastError();
	if (error ~= 0) then
		bContinue = false;
		fDebug("  ERROR: OS specific printer driver files copy to "..pathNetworkforOS.." failed with error: "..error);
	end
end

--[[ 
============================================================================================
 Install the legacy custom topcoat bitmaps if they are not present.
============================================================================================
 ]]--
if bContinue then
	fDebug("  INFO: Installing custom topcoat and erase bitmaps...");
	local pathCustomFolder = "C:\\CardPrinter"; 	-- location of custom files based on SP legacy location.
	local customBitmapFileNameTable = {"CRDPortTopcoatFront.bmp", "CRDPortTopcoatBack.bmp","CRDLandTopcoatFront.bmp", "CRDLandTopcoatBack.bmp", "CRDLandEraseFront.bmp", "CRDPortEraseFront.bmp"};
	
	if not(Folder.DoesExist(pathCustomFolder)) then
		fDebug("  Creating custom topcoat bitmaps folder at: "..pathCustomFolder.."...");
		Folder.Create(pathCustomFolder);
		error = Application.GetLastError();
		if (error ~= 0) then
			fDebug("  ERROR: Could not create folder.  Error: "..error);
		end
		-- TO DO: Check for error and report
	end
	
	-- loop through the image files and copy them if they do not exist
	local fileCount = Table.Count(customBitmapFileNameTable);
	local loopCounter = 1;
	repeat
		local strFileName = customBitmapFileNameTable[loopCounter];
		--  ******************************************************************************************
		--  Detect if the custom image file is installed.  If it is leave it and do not overwrite it.
		--  *******************************************************************************************
		if not (File.DoesExist(pathCustomFolder.."\\"..strFileName)) then
			fDebug("  Installing custom topcoat bitmap file: "..strFileName.."...");
			File.Copy(_SourceFolder.."\\AutoPlay\\Images\\"..strFileName, pathCustomFolder.."\\"..strFileName, false, false, false, false, nil); 
			error = Application.GetLastError();
			if (error ~= 0) then
				fDebug("  ERROR: Could not copy file.  Error: "..error);
			end
		else
			fDebug("  INFO: Custom bitmap file: "..strFileName.." detected.  Use existing file.");
		end
		loopCounter = loopCounter + 1;
	until loopCounter > fileCount
end

-- Send a signal back to the main script using the bContinueInstall flag.
if (bContinue) then
	bContinueInstall = true;
	fDebug("  SUCCESS: Driver files copied.");
else
	bContinueInstall = false;
	fDebug("  FAIL: Driver files not copied.");
end

bScriptRanToCompletion = true;
fDebug("[Script Exit]: Copy_Driver_Files \r\n");