--[[
********************************************************************
Determine if we offer to install the XPS Card Printer Plug-in during driver installation.

Last Modified: 6-21-2013
Last Modified by: Ken Fischer
Driver Version of last mod: v5.0

NOTES:
Branding: This script is brand aware.

Global variables used by this script: None

Global variables set by this script: globalOfferPluginInstall
************************************************************************* 
--]]

fDebug(" SCRIPT: Offer_Plugin_Install.lua \r\nScript started. ");

--[[
Installed = COM registered and file exists 
]]--
function GetPrintServerFile()
	local PRINT_SERVER_CLASS_ID = "{39826198-6E77-4D72-8757-FDFAF628AE85}";
	local fullPrintServerFilePath = Registry.GetValue(0, "CLSID\\"..PRINT_SERVER_CLASS_ID.."\\InprocServer32","",false);
	if((Application.GetLastError()) ~= 0)then 	-- Registry value was not found. Print Server not installed.	
		return nil;
	end
	if(not(File.DoesExist(fullPrintServerFilePath))) then
		return nil;
	end
	return fullPrintServerFilePath;
end

--[[
The Print Server folder is where we will look for the OEM skin file.
]]--
function GetPrintServerFolder(fullPrintServerFilePath)
	local printServerSplitPath = String.SplitPath(fullPrintServerFilePath);
	local driveLetter = printServerSplitPath.Drive;
	if (String.Length(driveLetter) < 1) then
		return nil
	end
	local folderPath = printServerSplitPath.Folder;
	if (String.Length(folderPath) < 1) then
		return nil
	end
	local printServerFolderPath = String.Concat(driveLetter, folderPath);
	fDebug("  Print Server folder: "..printServerFolderPath);
	return printServerFolderPath;
end

--[[
If this is an OEM install, the build number info will change.
]]--
function IsPolaroidInstall()
	local POLAROID_OEM_TAG = "OEM01";
	local OEMInstallTag = String.Upper(INIFile.GetValue(_SourceFolder.."\\Autoplay\\DXP01Install.ini", "Flags", "oem"));
	fDebug("  OEM install tag: "..OEMInstallTag);
	if (OEMInstallTag == POLAROID_OEM_TAG) then
		return true;
	end
	return false;
end

--[[
If this is an OEM install, check that this is a Polaroid branded ID application.
]]--
function IsPrintServerPolaroidBrand(printServerFolderPath)
	local POLAROID_OEM_RESOURCE_DLL = "skin01.dll";
	if (not (File.DoesExist(printServerFolderPath.."\\"..POLAROID_OEM_RESOURCE_DLL))) then
		fDebug("  OEM01 ID Application not detected. ");
		return false;
	end	
	return true;	
end

--[[
Plug-in requires ID Works v6.5 to function.
  NOTE: The DLL version info has the form: BuildNumber.month.day.year
]]--
function IsMinVersion65(thisIsPolaroidInstall,fullPrintServerFilePath)
	local V65_BUILD_NUMBER = 848;
	if (thisIsPolaroidInstall) then
		V65_BUILD_NUMBER = 860;
	end
	local printServerVersion = File.GetVersionInfo(fullPrintServerFilePath);	
	local buildNumberDelimiterPosition = String.Find(printServerVersion.FileVersion,".",1,false); 
	local printServerBuildNumber = String.ToNumber(String.Left((printServerVersion.FileVersion),(buildNumberDelimiterPosition-1)));
	fDebug("  Print Server build number: "..printServerBuildNumber);
	if (not (printServerBuildNumber >= V65_BUILD_NUMBER)) then
		return false;
	end
	return true;
end

--[[
Determine if the XPS Card Printer Plug-in exists in the Print Server directory.
]]--
function IsPluginInstalled(printServerFolderPath)
local XPS_CARD_PRINTER_PLUGIN_FILE_NAME = "psxpscardprinterplugin.dll";
	if (not (File.DoesExist(printServerFolderPath.."\\"..XPS_CARD_PRINTER_PLUGIN_FILE_NAME))) then
		return false;
	end
	return true;
end

--[[
Dtermine if the installed plug-in is older and a candidate for update.
]]--
function IsInstalledPluginOlder(thisIsPolaroidInstall,printServerFolderPath)
	local INSTALL_INI_FILE = _SourceFolder.."\\Autoplay\\DXP01Install.ini";
	local INSTALL_INI_FILE_PLUGIN_SECTION_HEADER = "Plugin";	
	local XPS_CARD_PRINTER_PLUGIN_FILE_NAME = "psxpscardprinterplugin.dll";
	local INSTALL_INI_FILE_PLUGIN_BUILD_KEY = nil;
	if (thisIsPolaroidInstall) then
		INSTALL_INI_FILE_PLUGIN_BUILD_KEY = "oem01build";
	else
		INSTALL_INI_FILE_PLUGIN_BUILD_KEY = "build";
	end		
	local installCandidatePluginBuildNumber = 0;
	local currentPluginBuildNumber = 0;
	
	--get build number from install media
	if (not (File.DoesExist(INSTALL_INI_FILE))) then
		return false;
	end
	installCandidatePluginBuildNumber = String.ToNumber(INIFile.GetValue(INSTALL_INI_FILE, INSTALL_INI_FILE_PLUGIN_SECTION_HEADER, INSTALL_INI_FILE_PLUGIN_BUILD_KEY));
	fDebug("  Media plug-in build number: "..installCandidatePluginBuildNumber);
	if (installCandidatePluginBuildNumber < 1) then
		return false;
	end
	
	-- get installed plug-in build number. relies on the fact that the build number is the first element in the version string o fthe DLL.
	local currentPluginVersion = File.GetVersionInfo(printServerFolderPath.."\\"..XPS_CARD_PRINTER_PLUGIN_FILE_NAME);	
	local buildNumberDelimiterPosition = String.Find(currentPluginVersion.FileVersion, ".", 1, false);
	local currentPluginBuildNumber = String.ToNumber(String.Left((currentPluginVersion.FileVersion),(buildNumberDelimiterPosition-1)));
	fDebug("  Installed plug-in build number: "..currentPluginBuildNumber);
	if ( currentPluginBuildNumber < 1) then
		return false;
	end	
	
	-- test if install media build is newer.
	if (currentPluginBuildNumber >= installCandidatePluginBuildNumber) then
		return false;
	end
	return true;
end

--[[
==================================================================== 
Call the functions and set the variables used by the 
install script to determine what options to offer.
====================================================================
]]--
-- Initialze global variable used by calling app
globalOfferPluginInstall = false;

-- Initialize local script variables
local fullPrintServerFilePath = nil;
local printServerFolderPath = nil;
local thisIsPolaroidInstall = false;

-- Execute the script functions

fullPrintServerFilePath = GetPrintServerFile();
if (fullPrintServerFilePath == nil) then
	fDebug("  Print Server not found. \r\n Script return.");
	return;
end 

printServerFolderPath = GetPrintServerFolder(fullPrintServerFilePath);
if (printServerFolderPath == nil) then
	fDebug("  Print Server folder not found. \r\n Script return.");
	return;
end 

thisIsPolaroidInstall = IsPolaroidInstall();
if (thisIsPolaroidInstall and (not IsPrintServerPolaroidBrand(printServerFolderPath))) then
	fDebug("  Print Server found. Wrong brand. \r\n Script return.");
	return;
end
	
if (not IsMinVersion65(thisIsPolaroidInstall,fullPrintServerFilePath)) then
	fDebug("  Print Server found. Version too old. \r\n Script return.");
	return;
end

if (not IsPluginInstalled(printServerFolderPath)) then
	globalOfferPluginInstall = true;
	fDebug("   **Offer install**. Print Server found. Plug-in not found. \r\n Script return. ");
	return;
end

if (IsInstalledPluginOlder(thisIsPolaroidInstall,printServerFolderPath)) then
	globalOfferPluginInstall = true;
	fDebug("   **Offer install**. Older plug-in found. \r\n Script return. ");
	return;
end

fDebug("  Print Server found. Same or newer plug-in installed. \r\n Script return. ");