--[[ 
====================================================================================
Script to determine if the DPX01 USB driver is installed.
Assumes that finding the Belcarra driver in the system32\drivers folder means the 
USB driver is installed.

Returns two booleans:
IsUSBDriverInstalled - tells calling function if the USBLAN driver is installed
IsUSBDriverUpdateNeeded - tells calling function if installed USBLAN driver needs to be updated
====================================================================================
]]--

fDebug("[Script Start}: ISUSBDriverInstalled");
IsUSBDriverInstalled = false; -- returned to the calling script
IsUSBDriverUpdateNeeded = false;  -- returned to the calling script
local strInstalledBelcarraVersion = "";

--[[***************************************************************
Determine if USBLAN driver is installed.
If it is found, get its file cversion.
*******************************************************************]]--
if (System.Is64BitOS()) then -- if 64 bit OS turn off redirection
  DLL.CallFunction("kernel32.dll", "Wow64DisableWow64FsRedirection", "\""..Application.GetWndHandle().."\"", DLL_RETURN_TYPE_LONG, DLL_CALL_STDCALL);
end

local strDriverFileToCheck = _SystemFolder.."\\Drivers\\dxp01bdblan.sys";
fDebug("  Looking for file: "..strDriverFileToCheck.."...");
if (File.DoesExist(strDriverFileToCheck)) then
	fDebug("  INFO: USBLAN driver file found.");
	IsUSBDriverInstalled = true;
	local tVersionInfo = File.GetVersionInfo(_SystemFolder.."\\drivers\\dxp01bdblan.sys");
	-- Get the file version
	local nError = Application.GetLastError();
	if (nError ~= 0) then
		fDebug("ERROR: Installed USBLAN driver file version check failed wih error: "..nError);
		IsDriverUpdatable = false; -- set this to false to force the installer to remove the installed driver.
		return;
	end
	strInstalledBelcarraVersion = tVersionInfo.FileVersion;
	fDebug("  INFO: Installed USBLAN driver file version: "..strInstalledBelcarraVersion);
else
	fDebug("  INFO: No USB driver file found.");
end

if (System.Is64BitOS()) then -- if 64 bit OS turn redirection back on.
  DLL.CallFunction("kernel32.dll", "Wow64RevertWow64FsRedirection", "\""..Application.GetWndHandle().."\"" , DLL_RETURN_TYPE_LONG, DLL_CALL_STDCALL);
end

--[[***************************************************************
If USBLAN driver was found, then get th OS specific version of the Belcarra driver
from the install media and determine if it is newer than the installed version.  If it is
set the global variable IsUSBDriverUpdateNeeded to true.
*******************************************************************]]--

if (IsUSBDriverInstalled)then
	local OSTypeFolder = "";
	if (System.Is64BitOS()) then
	 	OSTypeFolder = "AMD64";
	else
		OSTypeFolder = "i386";
	end
	local OSVersionTable = OSUtils.GetVersionElements();
	local pathToUSBLANDriver = "";
	-- any version less than 6.1 is prevented from installing so no need to check for OS's below Windows 7
	if (OSVersionTable.Major == 6 and OSVersionTable.Minor < 2) then -- Windows 7 and Server 2008      
		pathToUSBLANDriver = _SourceFolder.."\\USB\\Windows7\\"..OSTypeFolder.."\\dxp01bdblan.sys";
	elseif (OSVersionTable.Major == 6 and OSVersionTable.Minor > 1) then -- Windows 8, 8.1 and Server 2012         
		pathToUSBLANDriver = _SourceFolder.."\\USB\\Windows8\\"..OSTypeFolder.."\\dxp01bdblan.sys";
	elseif (OSVersionTable.Major == 10) then -- Windows 10
		pathToUSBLANDriver = _SourceFolder.."\\USB\\Windows10\\"..OSTypeFolder.."\\dxp01bdblan.sys";
	else -- assume Windows 10
		pathToUSBLANDriver = _SourceFolder.."\\USB\\Windows10\\"..OSTypeFolder.."\\dxp01bdblan.sys";
	end
	-- Get the USBLAN driver file version on the install media.
	local DriverFileInfo = File.GetVersionInfo(pathToUSBLANDriver);
	-- Get the last error code
	local nError = Application.GetLastError();
	if nError ~= 0 then
		fDebug("ERROR: Install media USBLAN driver file version check failed wih error: "..nError);
		IsDriverUpdatable = false; -- set this to false to force the installer to remove the installed driver.
		return;
	end
	local strMediaUSBDriverVersion = DriverFileInfo.FileVersion;
	fDebug("  INFO: Media USBLAN driver version: "..strMediaUSBDriverVersion);
	

-- compare the Belcarra versions to determine if installed version needs to be updated.
	-- code below decomposes the version string into 3 numbers: Major Version, Minor Version and Sub-Minor Version
	-- the fourth number in the version string ( a repeating build number) are ignored.
	
	-- Get the version numbers for the installed driver.
	-- Major Version
	local VersionDelimiter = String.Find(strInstalledBelcarraVersion, ".", 1, false); -- look for the first decimal point. The major version will be to the left of this.
	local strMajorVersion = String.Left(strInstalledBelcarraVersion,(VersionDelimiter-1)); -- extract the major version number from the version string.
	local InstalledMajorVersion = String.ToNumber(strMajorVersion);
	local VersionStringLength = String.Length(strInstalledBelcarraVersion);
	strInstalledBelcarraVersion = String.Right(strInstalledBelcarraVersion, (VersionStringLength-VersionDelimiter));
	-- Minor Version
	local VersionDelimiter = String.Find(strInstalledBelcarraVersion, ".", 1, false); -- look for the first decimal point. The minor version will be to the left of this.
	local strMinorVersion = String.Left(strInstalledBelcarraVersion,(VersionDelimiter-1)); -- extract the minor version number from the version string.
	local InstalledMinorVersion = String.ToNumber(strMinorVersion);
	VersionStringLength = String.Length(strInstalledBelcarraVersion);
	strInstalledBelcarraVersion = String.Right(strInstalledBelcarraVersion, (VersionStringLength-VersionDelimiter));
	--Build Number
	local VersionDelimiter = String.Find(strInstalledBelcarraVersion, ".", 1, false); -- look for the first decimal point. The sub-minor version will be to the left of this.
	local strSubMinorVersion = String.Left(strInstalledBelcarraVersion,(VersionDelimiter-1)); -- extract the sub-minor version number from the version string.
	local InstalledSubMinorVersion = String.ToNumber(strSubMinorVersion);

	fDebug("  INFO: system USBLAN driver version (used for comparison): "..InstalledMajorVersion.."."..InstalledMinorVersion.."."..InstalledSubMinorVersion);
	-- 
	-- Get the version numbers for the media driver.
	-- Major Version
	local VersionDelimiter = String.Find(strMediaUSBDriverVersion, ".", 1, false); -- look for the first decimal point. The major version will be to the left of this.
	local strMajorVersion = String.Left(strMediaUSBDriverVersion,(VersionDelimiter-1)); -- extract the major version number from the version string.
	local MediaMajorVersion = String.ToNumber(strMajorVersion);
	local VersionStringLength = String.Length(strMediaUSBDriverVersion);
	strMediaUSBDriverVersion = String.Right(strMediaUSBDriverVersion, (VersionStringLength-VersionDelimiter));
	-- Minor Version
	local VersionDelimiter = String.Find(strMediaUSBDriverVersion, ".", 1, false); -- look for the first decimal point. The minor version will be to the left of this.
	local strMinorVersion = String.Left(strMediaUSBDriverVersion,(VersionDelimiter-1)); -- extract the minor version number from the version string.
	local MediaMinorVersion = String.ToNumber(strMinorVersion);
	local VersionStringLength = String.Length(strMediaUSBDriverVersion);
	strMediaUSBDriverVersion = String.Right(strMediaUSBDriverVersion, (VersionStringLength-VersionDelimiter));
	-- Build Number
	local VersionDelimiter = String.Find(strMediaUSBDriverVersion, ".", 1, false); -- look for the first decimal point. The sub-minor version will be to the left of this.
	local strSubMinorVersion = String.Left(strMediaUSBDriverVersion,(VersionDelimiter-1)); -- extract the sub-minor version number from the version string.
	local MediaSubMinorVersion = String.ToNumber(strSubMinorVersion);

	fDebug("  INFO: Installer USBLAN driver version (used for comparison): "..MediaMajorVersion.."."..MediaMinorVersion.."."..MediaSubMinorVersion);
--
-- Start comparison to determine if update needed.
	if (InstalledMajorVersion < MediaMajorVersion) then
		IsUSBDriverUpdateNeeded = true;
		fDebug("  INFO: Installed USBLAN driver requires updating. Major version difference.");
	end
	if (InstalledMajorVersion == MediaMajorVersion) and (InstalledMinorVersion < MediaMinorVersion) then
		IsUSBDriverUpdateNeeded = true;
		fDebug("  INFO: Installed USBLAN driver requires updating. Minor version difference.");
	end
	if (InstalledMajorVersion == MediaMajorVersion) and (InstalledMinorVersion == MediaMinorVersion) and (InstalledSubMinorVersion < MediaSubMinorVersion) then
		IsUSBDriverUpdateNeeded = true;
		fDebug("  INFO: Installed USBLAN driver requires updating. Minor version difference.");
	end
	if not(IsUSBDriverUpdateNeeded) then
		fDebug("  INFO: Installed USBLAN driver does not require updating.");
	end	
else
	fDebug("  INFO: No USBLAN driver installed.");
end

bScriptRanToCompletion = true;
fDebug("[Script Exit]: IsUSBDriverInstalled \r\n");